#!/bin/sh
#
# ModemManager Auto-Reconnect Script with Logging
# ------------------------------------------------
# Monitors modem connectivity via mmcli and automatically reconnects
# if the network connection is lost.
#
# Logs:
#   - Printed to stdout (if run manually)
#   - Sent to syslog (tag: mm-auto-reconnect)
#
# Usage:
#   ./mm-auto-reconnect.sh [MODEM_INDEX]
# Example:
#   ./mm-auto-reconnect.sh 0
#

MODEM_INDEX="${1:-0}"        # Default modem: 1
MDEV="wwan0"

source /rw/wan_apn.conf

SLEEP_TIME=10           # Seconds between checks
LOG_TAG="mm-reconnect"  # Syslog tag

log() {
   local msg="[$(date '+%Y-%m-%d %H:%M:%S')] $*"
   echo "$msg" >> /tmp/mm.txt
   logger -t "$LOG_TAG" "$*"
}

log "Starting ModemManager auto-reconnect (modem=$MODEM_INDEX, apn=$APN)"

while true; do
    sleep "$SLEEP_TIME"

    # Check if modem exists
    if ! mmcli -m "$MODEM_INDEX" &>/dev/null; then
        log "⚠️  Modem $MODEM_INDEX not found. Retrying in $SLEEP_TIME sec..."
        continue
    fi

    # Get modem state
    STATE=$(mmcli -m "$MODEM_INDEX" -K | awk -F': ' '/modem.generic.state\s/ {print $2}')

    if [ "$STATE" == "connected" ]; then
        log "✅ Modem $MODEM_INDEX connected."
    else
        log "❌ Modem $MODEM_INDEX disconnected (state=$STATE). Attempting connect..."

        if [ -e /var/run/udhcpc.$MDEV ]; then
            log "Release DHCP lease for $MDEV"
            kill -USR2 $(cat /var/run/udhcpc.$MDEV)
        fi

        source /rw/wan_apn.conf
        if mmcli -m "$MODEM_INDEX" --simple-connect="$APN" &>/dev/null; then
            log "🔁 Successfully reconnected to APN '$APN'."
            if [ -e /var/run/udhcpc.$MDEV ]; then
               log "Renew DHCP lease for $MDEV"
               kill -USR1 $(cat /var/run/udhcpc.$MDEV)
            else
               log "Start DHCP for $MDEV"
               udhcpc -b -i $MDEV -t 4 -T 4 -p /var/run/udhcpc.$MDEV
            fi
        else
            log "💥 Reconnect attempt failed."
        fi
    fi
done
